/******************************************************************************
 *
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2010 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
 **************************************************************************//*!
 *
 * @file phdc_app.c
 *
 * @author
 *
 * @version
 *
 * @date May-28-2009
 *
 * @brief When the device is connected using continua manager it comes into
 *        operating state and the device specialization is Weighing Scale.
 *        Measurements can be sent by pressing PTG1 button on the demo board.
 *        Device specialization can be changed by pressing PTG3.
 *****************************************************************************/

/******************************************************************************
 * Includes
 *****************************************************************************/
#include "hidef.h"          /* for EnableInterrupts macro */
#include "derivative.h"     /* include peripheral declarations */
#include "types.h"          /* User Defined Data Types */
#include "phdc_app.h"       /* PHDC Application Header File */
#include "phd_com_model.h"  /* IEEE11073 Header File */
#include "phd_device_spec.h"
#include "realtimercounter.h"

#include "PulseOximeter.h"  /* Pulse Oximeter Routines */
#include "SwTimer.h"        /* Software Timer Routines */

#include "TWR_K50_UI.h"     /* LEDs and Buttons definitions */
#include "ADC.h"            /* ADC Handling */
#include "OPAMP.h"          /* OpAmp and TRIAMP Handling */

/* skip the inclusion in dependency statge */
#ifndef __NO_SETJMP
   #include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>			

#if (defined _MCF51MM256_H) || (defined _MCF51JE256_H)
#include "exceptions.h"
#endif
/*****************************************************************************
 * Constant and Macro's 
 *****************************************************************************/

/*****************************************************************************
 * Global Functions Prototypes
 *****************************************************************************/
void TestApp_Init(void);
extern void display_led(uint_8 val);
/****************************************************************************
 * Global Variables 
 ****************************************************************************/
extern uint_8 g_phd_selected_configuration;
/*****************************************************************************
 * Local Types
 *****************************************************************************/
typedef enum
{
 NO_MEASUREMENT,
 SPO2_MEASUREMENT
}MEAS_TYPES;

typedef enum
{
  DISABLE,
  ENABLE
}BOARD_STATUS;

/*****************************************************************************
 * Local Functions Prototypes
 *****************************************************************************/
static void USB_App_Callback(uint_8 controller_ID, uint_8 event_type);
static void App_Display(uint_8 val);
#ifdef TIMER_CALLBACK_ARG
static void SelectTimerCallback(void * arg);
#else
static void SelectTimerCallback(void);
#endif
void vfnAFE_Control(uint_8 u8Status);
void vfnSpO2_AFE_Init (void);
/*****************************************************************************
 * Local Variables
 *****************************************************************************/
static uint_8 event=APP_PHD_UNINITIALISED;

uint_8 g_app_timer = INVALID_TIMER_VALUE;
uint_8 ActualMeasurement = NO_MEASUREMENT;

/*****************************************************************************
 * Local Functions
 *****************************************************************************/
/******************************************************************************
 *
 *    @name        USB_App_Callback
 *
 *    @brief       This function handles the callback
 *
 *    @param       controller_ID    : Controller ID
 *    @param       event_type       : Value of the event
 *
 *    @return      None
 *
 *****************************************************************************
 * This function is called from the lower layer whenever an event occurs.
 * This sets a variable according to the event_type
 *****************************************************************************/
static void USB_App_Callback(
      uint_8 controller_ID, /* [IN] Controller ID */
      uint_8 event_type     /* [IN] Value of the event */
)
{
    UNUSED (controller_ID)
    /* Remove Timer if any */
    if(g_app_timer != INVALID_TIMER_VALUE)
    {
        (void)RemoveTimerQ(g_app_timer);
        g_app_timer = INVALID_TIMER_VALUE;
    }
    switch(event_type)
    {
        case APP_PHD_UNINITIALISED:
        case APP_PHD_INITIALISED:
		    App_Display((uint_8)g_phd_selected_configuration);

        case APP_PHD_CONNECTED_TO_HOST:
            if (ActualMeasurement == NO_MEASUREMENT)
            {
             ActualMeasurement = SPO2_MEASUREMENT;
             vfnAFE_Control(ENABLE);
             SpO2_DiagnosticModeStartMeasurement();
            }
            
        case APP_PHD_ERROR:
            scanReportNo = 0; /* Initialize with 0 each time the Agent is associated to the manager */
            event = event_type;
            break;
            
        case APP_PHD_MEASUREMENT_SENT:
            event = APP_PHD_CONNECTED_TO_HOST;
            break;
            
        case APP_PHD_DISCONNECTED_FROM_HOST:
           if(ActualMeasurement == SPO2_MEASUREMENT)
            {
             ActualMeasurement = NO_MEASUREMENT;
             vfnAFE_Control(DISABLE);
             SpO2_DiagnosticModeStopMeasurement();
            }

            
        case APP_PHD_ASSOCIATION_TIMEDOUT:
            event = APP_PHD_INITIALISED;
            break;
        
    }   
    return;
}

/******************************************************************************
 *
 *   @name        TestApp_Init
 *
 *   @brief       This function is the entry for the PHDC Multi Specialization
 *                Application 
 *
 *   @param       None
 *
 *   @return      None
 *
 *****************************************************************************
 * This function starts the PHDC (Multi Specialization demo) application
 *****************************************************************************/
void TestApp_Init(void)
{
    uint_8 error;

    DisableInterrupts;
	#if (defined _MCF51MM256_H) || (defined _MCF51JE256_H)
     usb_int_dis();
    #endif	

    /* Initialize the USB interface */
    error = PHD_Transport_Init(CONTROLLER_ID, USB_App_Callback);

    EnableInterrupts;
	#if (defined _MCF51MM256_H) || (defined _MCF51JE256_H)
     usb_int_en();
    #endif	
     
     /* LEDs and Buttons Initialization */
    GPIOS_CLOCK_INIT;                                                           //Enables clock on all GPIOs
    INIT_ALL_LEDS;                                                              //Initialize both LEDs on the board
    INIT_ALL_PBS;                                                               //Initialize both Push Buttons on the board
    
    /* Initialize the Analog Front End */
    SwTimer_Init();                                                             //Initializes software timer
    vfnSpO2_AFE_Init();                                                         //Initializes peripherals required by MED-SPO2 AFE
    
}

/******************************************************************************
 *
 *   @name        TestApp_Task
 *
 *   @brief       Application task function. It is called from the main loop
 *
 *   @param       None
 *
 *   @return      None
 *
 *****************************************************************************
 * Application task function. It is called from the main loop
 *****************************************************************************/
void TestApp_Task(void)
{
        
        switch (event)
        {
            case APP_PHD_INITIALISED:
                {
                    /* 
                        Start a timer so that the user has enough time to 
                        select the device specialization 
                    */
                    TIMER_OBJECT TimerObject;
                    TimerObject.msCount = SELECT_TIMEOUT;
                    TimerObject.pfnTimerCallback = SelectTimerCallback;
                 
                    event = APP_PHD_SELECT_TIMER_STARTED;
                    g_app_timer = AddTimerQ(&TimerObject);
                }
                break;
            case APP_PHD_DISCONNECTED_FROM_HOST:
                /* 
                    transition to initialised state so the association 
                    procedure can start again 
                */
                event = APP_PHD_INITIALISED;
                break;

            case APP_PHD_MEASUREMENT_SENT:
                /* 
                    enters here each time we receive a response to the
                    measurements sent 
                */
                event = APP_PHD_CONNECTED_TO_HOST;
                break;
            case APP_PHD_SELECT_TIMER_OFF:
                /* 
                    Start the association procedure once the select timer 
                    fires 
                */
                event = APP_PHD_INITIATED;
                /* connect to the manager */
                PHD_Connect_to_Manager(CONTROLLER_ID);                  
                break;
             
                  
          default:
              break;
    
        }
        
        SpO2_PeriodicTask();
        SwTimer_PeriodicTask();

}

/******************************************************************************
 *
 *   @name        App_Display
 *
 *   @brief       This function displays the leds on the demo board 
 *                corressponding to the device specialization currently running
 *
 *   @param       val   : Value to display
 *
 *   @return      None
 *
 *****************************************************************************
 * This funtions displays the LEDs on the board so that user knows which 
 * device specialization is active
 *****************************************************************************/
static void App_Display (
    uint_8 val      /* [IN] Value to display */
)
{
#ifndef _MC9S08JS16_H
    /* LED Display Code */
    display_led((uint_8)(0x1<<val));
#else
    /* JS16 Specific LED Display Code */
    display_led(val);
#endif
}
/******************************************************************************
 *
 *   @name        SelectTimerCallback
 *
 *   @brief       This function is called when the timer used for selecting the
 *                the device specialization fires
 *
 *   @param       arg   : Argument passed by Timer ISR (optional)
 *
 *   @return      None
 *
 *****************************************************************************
 * This function when called initiates the association procedure for the 
 * selected device specialization
 *****************************************************************************/
#ifdef TIMER_CALLBACK_ARG
static void SelectTimerCallback(void * arg)
#else
static void SelectTimerCallback(void)
#endif
{
#ifdef TIMER_CALLBACK_ARG
    UNUSED (arg)
#endif
    g_app_timer = INVALID_TIMER_VALUE;
    event = APP_PHD_SELECT_TIMER_OFF;
}

/******************************************************************************
 *
 *   @name        vfnAFE_Control(u8Status)
 *
 *   @brief       This function controls the AFE status (Enabled Disabled) by
 *                powering on or off the board
 *
 *   @param       u8Status
 *
 *   @return      Void
 *****************************************************************************/

void vfnAFE_Control(uint_8 u8Status)
{
 PORTC_PCR14 = PORT_PCR_MUX(1);                                               //PTC14 (AFE Enable PIN) as GPIO
 GPIOC_PDDR |= (1<<14);                                                       //PTC14 as Output
 
 if(u8Status == ENABLE)                                                       //If command is ENABLE
   GPIOC_PCOR |= (1<<14);                                                     //Clear pin to enable power MOSFET
 
 if(u8Status == DISABLE)                                                      //If command is DISABLE
   GPIOC_PSOR |= (1<<14);                                                     //Set pin to disable power MOSFET
}

/******************************************************************************
 *
 *   @name        vfnSpO2_AFE_Init
 *
 *   @brief       This function initializes all the required peripherals for
 *                MED-SpO2 
 *
 *   @param       Void
 *
 *   @return      Void
 *
 *****************************************************************************/

void vfnSpO2_AFE_Init (void)
{
 vfnAFE_Control(ENABLE);                                                        //Enables power on MED-SPO2 AFE
 
 opamp1_gp_mode();                                                              //Enables OpAmp0 as General Purpose OpAmp
 opamp2_gp_mode();                                                              //Enables OpAmp1 as General Purpose OpAmp
 TRIAMP1C0_HighMode();                                                          //Enables TRIAMP0 as General Purpose OpAmp
 TRIAMP2C0_HighMode();                                                          //Enables TRIAMP1 as General Purpose OpAmp
 
 ADC0_Init12b();                                                                //Initialize ADC0 12-bit Resolution
 ADC1_Init12b();                                                                //Initialize ADC1 12-bit Resolution
 
 SpO2_Init();                                                                   //SpO2 Timers Initialization Routine
 
 vfnAFE_Control(DISABLE);                                                       //Disables power on MED-SPO2 AFE
}